<?php
require_once 'config/database.php';
include 'includes/header.php';

$database = new Database();
$db = $database->getConnection();

// Obtener categorías y proveedores para los selects
try {
    $stmt = $db->prepare("SELECT * FROM categorias WHERE activo = 1 ORDER BY nombre");
    $stmt->execute();
    $categorias = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $stmt = $db->prepare("SELECT * FROM proveedores WHERE activo = 1 ORDER BY nombre");
    $stmt->execute();
    $proveedores = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch(PDOException $exception) {
    echo "Error: " . $exception->getMessage();
}
?>

<div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
    <h1 class="h2"><i class="fas fa-box me-2"></i>Gestión de Productos</h1>
    <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#modalProducto">
        <i class="fas fa-plus me-1"></i>Nuevo Producto
    </button>
</div>

<!-- Filtros -->
<div class="card mb-4">
    <div class="card-body">
        <div class="row">
            <div class="col-md-4">
                <input type="text" class="form-control" id="filtroNombre" placeholder="Buscar por nombre...">
            </div>
            <div class="col-md-3">
                <select class="form-select" id="filtroCategoria">
                    <option value="">Todas las categorías</option>
                    <?php foreach($categorias as $categoria): ?>
                        <option value="<?php echo $categoria['id']; ?>"><?php echo htmlspecialchars($categoria['nombre']); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-3">
                <select class="form-select" id="filtroStock">
                    <option value="">Todos los productos</option>
                    <option value="bajo">Stock bajo</option>
                    <option value="sin_stock">Sin stock</option>
                </select>
            </div>
            <div class="col-md-2">
                <button type="button" class="btn btn-outline-primary w-100" onclick="cargarProductos()">
                    <i class="fas fa-search"></i> Buscar
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Tabla de productos -->
<div class="card">
    <div class="card-body">
        <div class="loading">
            <div class="spinner-border text-primary" role="status">
                <span class="visually-hidden">Cargando...</span>
            </div>
        </div>
        <div class="table-responsive">
            <table class="table table-hover" id="tablaProductos">
                <thead>
                    <tr>
                        <th>Código</th>
                        <th>Nombre</th>
                        <th>Categoría</th>
                        <th>Precio Compra</th>
                        <th>Precio Venta</th>
                        <th>Stock</th>
                        <th>Estado</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody id="cuerpoTablaProductos">
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal para agregar/editar producto -->
<div class="modal fade" id="modalProducto" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="tituloModal">Nuevo Producto</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="formProducto">
                <div class="modal-body">
                    <input type="hidden" id="productoId" name="id">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="codigoBarras" class="form-label">Código de Barras</label>
                                <input type="text" class="form-control" id="codigoBarras" name="codigo_barras">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="nombre" class="form-label">Nombre *</label>
                                <input type="text" class="form-control" id="nombre" name="nombre" required>
                            </div>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label for="descripcion" class="form-label">Descripción</label>
                        <textarea class="form-control" id="descripcion" name="descripcion" rows="3"></textarea>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="precioCompra" class="form-label">Precio de Compra *</label>
                                <input type="number" class="form-control" id="precioCompra" name="precio_compra" step="0.01" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="precioVenta" class="form-label">Precio de Venta *</label>
                                <input type="number" class="form-control" id="precioVenta" name="precio_venta" step="0.01" required>
                            </div>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label for="stockActual" class="form-label">Stock Actual</label>
                                <input type="number" class="form-control" id="stockActual" name="stock_actual" value="0">
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label for="stockMinimo" class="form-label">Stock Mínimo</label>
                                <input type="number" class="form-control" id="stockMinimo" name="stock_minimo" value="5">
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label for="stockMaximo" class="form-label">Stock Máximo</label>
                                <input type="number" class="form-control" id="stockMaximo" name="stock_maximo" value="1000">
                            </div>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="categoriaId" class="form-label">Categoría</label>
                                <select class="form-select" id="categoriaId" name="categoria_id">
                                    <option value="">Seleccionar categoría</option>
                                    <?php foreach($categorias as $categoria): ?>
                                        <option value="<?php echo $categoria['id']; ?>"><?php echo htmlspecialchars($categoria['nombre']); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="proveedorId" class="form-label">Proveedor</label>
                                <select class="form-select" id="proveedorId" name="proveedor_id">
                                    <option value="">Seleccionar proveedor</option>
                                    <?php foreach($proveedores as $proveedor): ?>
                                        <option value="<?php echo $proveedor['id']; ?>"><?php echo htmlspecialchars($proveedor['nombre']); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="unidadMedida" class="form-label">Unidad de Medida</label>
                                <select class="form-select" id="unidadMedida" name="unidad_medida">
                                    <option value="unidad">Unidad</option>
                                    <option value="kg">Kilogramo</option>
                                    <option value="gramo">Gramo</option>
                                    <option value="litro">Litro</option>
                                    <option value="ml">Mililitro</option>
                                    <option value="metro">Metro</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="fechaVencimiento" class="form-label">Fecha de Vencimiento</label>
                                <input type="date" class="form-control" id="fechaVencimiento" name="fecha_vencimiento">
                            </div>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label for="imagen" class="form-label">URL de Imagen</label>
                        <input type="url" class="form-control" id="imagen" name="imagen" placeholder="https://ejemplo.com/imagen.jpg">
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Guardar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
$(document).ready(function() {
    cargarProductos();
    
    // Filtros en tiempo real
    $('#filtroNombre').on('keyup', function() {
        cargarProductos();
    });
});

function cargarProductos() {
    showLoading();
    
    const filtros = {
        nombre: $('#filtroNombre').val(),
        categoria: $('#filtroCategoria').val(),
        stock: $('#filtroStock').val()
    };
    
    $.ajax({
        url: 'api/productos.php',
        method: 'GET',
        data: filtros,
        success: function(response) {
            const productos = JSON.parse(response);
            let html = '';
            
            productos.forEach(function(producto) {
                const stockClass = producto.stock_actual <= producto.stock_minimo ? 'text-danger' : 'text-success';
                const estadoBadge = producto.activo == 1 ? '<span class="badge bg-success">Activo</span>' : '<span class="badge bg-danger">Inactivo</span>';
                
                html += `
                    <tr>
                        <td>${producto.codigo_barras || 'N/A'}</td>
                        <td>
                            <div class="d-flex align-items-center">
                                <img src="${producto.imagen}" alt="${producto.nombre}" class="me-2" style="width: 40px; height: 40px; object-fit: cover; border-radius: 4px;">
                                <div>
                                    <strong>${producto.nombre}</strong>
                                    ${producto.descripcion ? '<br><small class="text-muted">' + producto.descripcion + '</small>' : ''}
                                </div>
                            </div>
                        </td>
                        <td>${producto.categoria_nombre || 'Sin categoría'}</td>
                        <td>Q${parseFloat(producto.precio_compra).toFixed(2)}</td>
                        <td>Q${parseFloat(producto.precio_venta).toFixed(2)}</td>
                        <td><span class="${stockClass}">${producto.stock_actual} ${producto.unidad_medida}</span></td>
                        <td>${estadoBadge}</td>
                        <td>
                            <button class="btn btn-sm btn-outline-primary" onclick="editarProducto(${producto.id})">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button class="btn btn-sm btn-outline-danger" onclick="eliminarProducto(${producto.id})">
                                <i class="fas fa-trash"></i>
                            </button>
                        </td>
                    </tr>
                `;
            });
            
            $('#cuerpoTablaProductos').html(html);
            hideLoading();
        },
        error: function() {
            showAlert('error', 'Error', 'No se pudieron cargar los productos');
            hideLoading();
        }
    });
}

$('#formProducto').on('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    const id = $('#productoId').val();
    const method = id ? 'PUT' : 'POST';
    
    $.ajax({
        url: 'api/productos.php',
        method: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function(response) {
            const result = JSON.parse(response);
            if (result.success) {
                showAlert('success', 'Éxito', result.message);
                $('#modalProducto').modal('hide');
                $('#formProducto')[0].reset();
                cargarProductos();
            } else {
                showAlert('error', 'Error', result.message);
            }
        },
        error: function() {
            showAlert('error', 'Error', 'No se pudo guardar el producto');
        }
    });
});

function editarProducto(id) {
    $.ajax({
        url: 'api/productos.php?id=' + id,
        method: 'GET',
        success: function(response) {
            const producto = JSON.parse(response);
            
            $('#productoId').val(producto.id);
            $('#codigoBarras').val(producto.codigo_barras);
            $('#nombre').val(producto.nombre);
            $('#descripcion').val(producto.descripcion);
            $('#precioCompra').val(producto.precio_compra);
            $('#precioVenta').val(producto.precio_venta);
            $('#stockActual').val(producto.stock_actual);
            $('#stockMinimo').val(producto.stock_minimo);
            $('#stockMaximo').val(producto.stock_maximo);
            $('#categoriaId').val(producto.categoria_id);
            $('#proveedorId').val(producto.proveedor_id);
            $('#unidadMedida').val(producto.unidad_medida);
            $('#fechaVencimiento').val(producto.fecha_vencimiento);
            $('#imagen').val(producto.imagen);
            
            $('#tituloModal').text('Editar Producto');
            $('#modalProducto').modal('show');
        }
    });
}

function eliminarProducto(id) {
    confirmAction('¿Eliminar producto?', 'Esta acción no se puede deshacer', function() {
        $.ajax({
            url: 'api/productos.php',
            method: 'DELETE',
            data: { id: id },
            success: function(response) {
                const result = JSON.parse(response);
                if (result.success) {
                    showAlert('success', 'Éxito', result.message);
                    cargarProductos();
                } else {
                    showAlert('error', 'Error', result.message);
                }
            }
        });
    });
}

// Limpiar formulario al cerrar modal
$('#modalProducto').on('hidden.bs.modal', function() {
    $('#formProducto')[0].reset();
    $('#productoId').val('');
    $('#tituloModal').text('Nuevo Producto');
});
</script>

<?php include 'includes/footer.php'; ?>