<?php
require_once 'config/database.php';
include 'includes/header.php';

$database = new Database();
$db = $database->getConnection();
?>

<div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
    <h1 class="h2"><i class="fas fa-shopping-cart me-2"></i>Punto de Venta</h1>
    <button type="button" class="btn btn-success" onclick="nuevaVenta()">
        <i class="fas fa-plus me-1"></i>Nueva Venta
    </button>
</div>

<div class="row">
    <!-- Panel de productos -->
    <div class="col-md-8">
        <div class="card">
            <div class="card-header">
                <h5><i class="fas fa-search me-2"></i>Buscar Productos</h5>
            </div>
            <div class="card-body">
                <div class="row mb-3">
                    <div class="col-md-8">
                        <input type="text" class="form-control" id="buscarProducto" placeholder="Buscar por nombre o código de barras...">
                    </div>
                    <div class="col-md-4">
                        <button type="button" class="btn btn-primary w-100" onclick="buscarProductos()">
                            <i class="fas fa-search"></i> Buscar
                        </button>
                    </div>
                </div>
                
                <div class="row" id="productosGrid">
                    <!-- Los productos se cargarán aquí -->
                </div>
            </div>
        </div>
    </div>
    
    <!-- Carrito de compras -->
    <div class="col-md-4">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5><i class="fas fa-shopping-cart me-2"></i>Carrito</h5>
                <button type="button" class="btn btn-sm btn-outline-danger" onclick="limpiarCarrito()">
                    <i class="fas fa-trash"></i>
                </button>
            </div>
            <div class="card-body">
                <div id="carritoItems" style="max-height: 400px; overflow-y: auto;">
                    <div class="text-center text-muted py-4">
                        <i class="fas fa-shopping-cart fa-3x mb-3"></i>
                        <p>El carrito está vacío</p>
                    </div>
                </div>
                
                <hr>
                
                <div class="d-flex justify-content-between mb-2">
                    <span>Subtotal:</span>
                    <span id="subtotal">Q0.00</span>
                </div>
                <div class="d-flex justify-content-between mb-2">
                    <span>Descuento:</span>
                    <span id="descuento">Q0.00</span>
                </div>
                <div class="d-flex justify-content-between mb-3">
                    <strong>Total:</strong>
                    <strong id="total">Q0.00</strong>
                </div>
                
                <div class="mb-3">
                    <label for="metodoPago" class="form-label">Método de Pago</label>
                    <select class="form-select" id="metodoPago">
                        <option value="efectivo">Efectivo</option>
                        <option value="tarjeta">Tarjeta</option>
                        <option value="transferencia">Transferencia</option>
                    </select>
                </div>
                
                <button type="button" class="btn btn-success w-100" onclick="procesarVenta()" id="btnProcesarVenta" disabled>
                    <i class="fas fa-cash-register me-1"></i>Procesar Venta
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Modal para agregar cantidad -->
<div class="modal fade" id="modalCantidad" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Agregar al Carrito</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="text-center mb-3">
                    <img id="imagenProductoModal" src="" alt="" class="img-fluid" style="max-height: 100px;">
                    <h6 id="nombreProductoModal" class="mt-2"></h6>
                    <p class="text-muted" id="precioProductoModal"></p>
                </div>
                <div class="mb-3">
                    <label for="cantidadProducto" class="form-label">Cantidad</label>
                    <input type="number" class="form-control" id="cantidadProducto" value="1" min="1" step="0.1">
                </div>
                <div class="mb-3">
                    <label for="descuentoProducto" class="form-label">Descuento (Q)</label>
                    <input type="number" class="form-control" id="descuentoProducto" value="0" min="0" step="0.01">
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                <button type="button" class="btn btn-primary" onclick="agregarAlCarrito()">Agregar</button>
            </div>
        </div>
    </div>
</div>

<script>
let carrito = [];
let productoSeleccionado = null;

$(document).ready(function() {
    buscarProductos();
    
    $('#buscarProducto').on('keypress', function(e) {
        if (e.which == 13) {
            buscarProductos();
        }
    });
});

function buscarProductos() {
    const termino = $('#buscarProducto').val();
    
    $.ajax({
        url: 'api/productos.php',
        method: 'GET',
        data: { nombre: termino },
        success: function(response) {
            const productos = JSON.parse(response);
            mostrarProductos(productos);
        }
    });
}

function mostrarProductos(productos) {
    let html = '';
    
    productos.forEach(function(producto) {
        if (producto.stock_actual > 0) {
            html += `
                <div class="col-md-6 mb-3">
                    <div class="card h-100 producto-card" onclick="seleccionarProducto(${producto.id})">
                        <img src="${producto.imagen}" class="card-img-top" alt="${producto.nombre}" style="height: 120px; object-fit: cover;">
                        <div class="card-body p-2">
                            <h6 class="card-title mb-1">${producto.nombre}</h6>
                            <p class="card-text mb-1">
                                <strong>Q${parseFloat(producto.precio_venta).toFixed(2)}</strong>
                            </p>
                            <small class="text-muted">Stock: ${producto.stock_actual}</small>
                        </div>
                    </div>
                </div>
            `;
        }
    });
    
    $('#productosGrid').html(html);
}

function seleccionarProducto(id) {
    $.ajax({
        url: 'api/productos.php?id=' + id,
        method: 'GET',
        success: function(response) {
            productoSeleccionado = JSON.parse(response);
            
            $('#imagenProductoModal').attr('src', productoSeleccionado.imagen);
            $('#nombreProductoModal').text(productoSeleccionado.nombre);
            $('#precioProductoModal').text('Q' + parseFloat(productoSeleccionado.precio_venta).toFixed(2));
            $('#cantidadProducto').val(1);
            $('#descuentoProducto').val(0);
            
            $('#modalCantidad').modal('show');
        }
    });
}

function agregarAlCarrito() {
    const cantidad = parseFloat($('#cantidadProducto').val());
    const descuento = parseFloat($('#descuentoProducto').val());
    
    if (cantidad <= 0 || cantidad > productoSeleccionado.stock_actual) {
        showAlert('error', 'Error', 'Cantidad no válida');
        return;
    }
    
    // Verificar si el producto ya está en el carrito
    const existeIndex = carrito.findIndex(item => item.id === productoSeleccionado.id);
    
    if (existeIndex !== -1) {
        carrito[existeIndex].cantidad += cantidad;
        carrito[existeIndex].descuento_unitario = descuento;
    } else {
        carrito.push({
            id: productoSeleccionado.id,
            nombre: productoSeleccionado.nombre,
            precio_unitario: parseFloat(productoSeleccionado.precio_venta),
            cantidad: cantidad,
            descuento_unitario: descuento,
            imagen: productoSeleccionado.imagen
        });
    }
    
    actualizarCarrito();
    $('#modalCantidad').modal('hide');
}

function actualizarCarrito() {
    let html = '';
    let subtotal = 0;
    let totalDescuento = 0;
    
    if (carrito.length === 0) {
        html = `
            <div class="text-center text-muted py-4">
                <i class="fas fa-shopping-cart fa-3x mb-3"></i>
                <p>El carrito está vacío</p>
            </div>
        `;
        $('#btnProcesarVenta').prop('disabled', true);
    } else {
        carrito.forEach(function(item, index) {
            const itemSubtotal = (item.precio_unitario - item.descuento_unitario) * item.cantidad;
            subtotal += itemSubtotal;
            totalDescuento += item.descuento_unitario * item.cantidad;
            
            html += `
                <div class="d-flex align-items-center mb-2 p-2 border rounded">
                    <img src="${item.imagen}" alt="${item.nombre}" class="me-2" style="width: 40px; height: 40px; object-fit: cover;">
                    <div class="flex-grow-1">
                        <small class="fw-bold">${item.nombre}</small><br>
                        <small class="text-muted">Q${item.precio_unitario.toFixed(2)} x ${item.cantidad}</small>
                    </div>
                    <div class="text-end">
                        <small class="fw-bold">Q${itemSubtotal.toFixed(2)}</small><br>
                        <button class="btn btn-sm btn-outline-danger" onclick="eliminarDelCarrito(${index})">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                </div>
            `;
        });
        $('#btnProcesarVenta').prop('disabled', false);
    }
    
    $('#carritoItems').html(html);
    $('#subtotal').text('Q' + (subtotal + totalDescuento).toFixed(2));
    $('#descuento').text('Q' + totalDescuento.toFixed(2));
    $('#total').text('Q' + subtotal.toFixed(2));
}

function eliminarDelCarrito(index) {
    carrito.splice(index, 1);
    actualizarCarrito();
}

function limpiarCarrito() {
    confirmAction('¿Limpiar carrito?', 'Se eliminarán todos los productos del carrito', function() {
        carrito = [];
        actualizarCarrito();
    });
}

function procesarVenta() {
    if (carrito.length === 0) {
        showAlert('error', 'Error', 'El carrito está vacío');
        return;
    }
    
    const venta = {
        items: carrito,
        metodo_pago: $('#metodoPago').val(),
        subtotal: parseFloat($('#subtotal').text().replace('Q', '')),
        descuento: parseFloat($('#descuento').text().replace('Q', '')),
        total: parseFloat($('#total').text().replace('Q', ''))
    };
    
    $.ajax({
        url: 'api/ventas.php',
        method: 'POST',
        data: JSON.stringify(venta),
        contentType: 'application/json',
        success: function(response) {
            const result = JSON.parse(response);
            if (result.success) {
                showAlert('success', 'Venta Procesada', 'Venta #' + result.numero_venta + ' procesada correctamente');
                carrito = [];
                actualizarCarrito();
                buscarProductos(); // Actualizar stock
            } else {
                showAlert('error', 'Error', result.message);
            }
        },
        error: function() {
            showAlert('error', 'Error', 'No se pudo procesar la venta');
        }
    });
}

function nuevaVenta() {
    limpiarCarrito();
    $('#buscarProducto').val('');
    buscarProductos();
}
</script>

<style>
.producto-card {
    cursor: pointer;
    transition: transform 0.2s ease;
}

.producto-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.15);
}
</style>

<?php include 'includes/footer.php'; ?>